/*
*******************************************************************************
*
* File:         physical.c
* Description:  
* Created:      Mon Oct 16 16:18:56 2000
* Modified:     Wed Jan 10 11:22:02 2001 by Henrik Liebau, BVS R&D
* Language:     C
* Package:      N/A
* Status:       Unpublished (Do Not Distribute)
*
* (C) Copyright 1997-2000, Agilent Technologies, All Rights Reserved.
*
*******************************************************************************
*/

/*------------------------------------------------------------------------
**
**	Copyright (c) 1997-2000, Agilent Technologies.  All rights reserved.
**
**	THIS IS UNPUBLISHED PROPRIETARY SOURCE CODE OF
**	Agilent Technologies Corp.
**	The copyright notice above does not evidence any
**	actual or intended publication of such source code.
**
**
**	Source: BVS, Agilent Technologies.      Tilmann Wendel
**
**
-------------------------------------------------------------------------*/

/*------------------------------------------------------------------------
 * functions for allocation of physical memory and mapping of
 * physical memory into virtual memory
 *------------------------------------------------------------------------
 */

#include <svpplatf.h>

#ifndef _DOS
#include <stdio.h>
#include <stddef.h>
#include <stdlib.h>

#ifdef _MSC_VER
#pragma warning( push, 3)
#endif

#include <windows.h>
#include <winioctl.h>

#ifdef _MSC_VER
#pragma warning( pop )
#endif

#include <physical.h>

// chris: needed for allocating buffers on 128 byte (ADB) boundaries
static const BYTE ADB_MASK=0x7f; // 127 bytes

HANDLE hDriver = INVALID_HANDLE_VALUE;  // keep driver open while the program
                                        // is running


#if _WIN32_WINNT >= 0x0500

/* forward declaration */
static BOOL
LoggedSetLockPagesPrivilege ( HANDLE hProcess,
                              BOOL bEnable);

#endif /* _WIN32_WINNT >= 0x0500 */

/* definition */
// print the error reported by GetLastError
static void
  print_err(char *fmt)
{
  DWORD lastErr;
  LPVOID lpMsgBuf;
  lastErr = GetLastError();
  FormatMessage(
    FORMAT_MESSAGE_ALLOCATE_BUFFER | FORMAT_MESSAGE_FROM_SYSTEM,
    NULL,
    lastErr,
    MAKELANGID(LANG_NEUTRAL, SUBLANG_DEFAULT),  // Default language
    (LPTSTR) & lpMsgBuf,
    0,
    NULL
    );
  printf(fmt, lpMsgBuf);
  LocalFree(lpMsgBuf);
}


// get the virtual address for a given physical address
// translation is done by calling the E2920 MEM driver
void *
  map_physical_address(
      unsigned long phys_addr_lo, 
      unsigned long phys_addr_hi,
      unsigned long length, 
      unsigned int bus_number, 
      unsigned int iospace)
{
  BOOL IoctlResult;

  DWORD cbReturned = 0;
  MPS_INFO ReturnStruct = {NULL, 0, 0};
  // 27.3.98 void *virt_addr = NULL;

  char * datebuf[DATE_BUFFER_SIZE];

  PHYSICAL_MEMORY_INFO pmi;
  pmi.InterfaceType = PCIBus;
  pmi.BusNumber = bus_number;
  pmi.BusAddress.HighPart = phys_addr_hi;
  pmi.BusAddress.LowPart = phys_addr_lo;
  pmi.Length = length;
  pmi.AddressSpace = iospace;   // 0=memory 1=I/O

  if (hDriver == INVALID_HANDLE_VALUE)
    hDriver = CreateFile(
      BASE_CREATE_FILE_NAME_PCI,
      GENERIC_WRITE | GENERIC_READ,
      0,
      NULL,
      OPEN_EXISTING,
      FILE_ATTRIBUTE_NORMAL,
      NULL);

  if (hDriver == INVALID_HANDLE_VALUE)
  {
    print_err("CreateFile failed: %s\n");
    goto end;
  }



  // make sure  you are talking to the right driver!!!!!!!!
  datebuf[0] = '\0';

  IoctlResult = DeviceIoControl(
    hDriver,                    /* Handle to device  */
    (DWORD) IOCTL_BEST_NTMEM_GET_VERSIONDATE,  /* IO Control code */
    NULL,                       /* Buffer to driver. */
    0,                          /* Length of buffer in bytes. */
    datebuf,                    /* Buffer from driver.  */
    DATE_BUFFER_SIZE,           /* Length of buffer in bytes. 27.3.98; was PVOID */
    &cbReturned,                /* Bytes placed in OutBuffer. 27.3.98; was cbReturned */
    NULL                        /* NULL means wait till op. completes. */
    );

  if (!IoctlResult)
  {
    printf("\n ************* Reading the version string failed. ************* \n\n"); 
    return NULL;
  }
  else
  {
    printf("The driver \"b_2kpci[_64].sys\"was made: %s.\n",datebuf); 
  }




  IoctlResult = DeviceIoControl(
    hDriver,                    /* Handle to device  */
    (DWORD) IOCTL_BEST_NTMEM_MAP_USER_PHYSICAL_MEMORY,  /* IO Control code */
    &pmi,                       /* Buffer to driver. */
    sizeof(pmi),                /* Length of buffer in bytes. */
    &ReturnStruct,              /* Buffer from driver.  */
    sizeof(MPS_INFO),           /* Length of buffer in bytes. 27.3.98; was PVOID */
    &cbReturned,                /* Bytes placed in OutBuffer. 27.3.98; was cbReturned */
    NULL                        /* NULL means wait till op. completes. */
    );

  // TODO; This is debug code
  // As soon as it has served its purpose this should be removed and
  // the b_ntmem.sys replaced.  In order to retrieve extended error codes
  // we "fool" NT into thinking that the call succeeded...otherwise it
  // zeroes out the return buffer.  
  // Thus the debug driver always returns OK and IoctlResult is always TRUE.

//  if (!IoctlResult)
  if(0xff != ReturnStruct.ntStatus)   // TO BE REMOVED.
  {
    // print_err("DeviceIoControl failed (MAP_USER_PHYSICAL_MEMORY):\n%s\n");
    printf("\n\nMap Physical Memory Failed.\n"); 
	printf("  Trying to map from %08x %08x for 0x%x bytes, bus %d, mem/io %d\n",
		phys_addr_hi,phys_addr_lo,length,bus_number,iospace);
    printf("  Extended Error Codes (please report to HP): %x, %x\n", 
        ReturnStruct.ntStatus, ReturnStruct.driverErr); 
    ReturnStruct.virt_addr = NULL;
  }

  /* keep driver open... 
  if (!CloseHandle(hDriver)) 
  { 
    printf("CloseHandle failed\n"); 
  } 
  */

end:
  return (ReturnStruct.virt_addr);  // 27.3.98 was virt_addr
}




// unmap a virtual address that was generated by map_physical_address
void
  unmap_physical_address(void *virt_addr)
{
  BOOL IoctlResult;
  DWORD cbReturned;
  if (hDriver == INVALID_HANDLE_VALUE)
    hDriver = CreateFile(
      BASE_CREATE_FILE_NAME_PCI,
      GENERIC_WRITE | GENERIC_READ,
      0,
      NULL,
      OPEN_EXISTING,
      FILE_ATTRIBUTE_NORMAL,
      NULL);

  if (hDriver == INVALID_HANDLE_VALUE)
  {
    print_err("CreateFile failed: %s\n");
    goto end;
  }

  IoctlResult = DeviceIoControl(
    hDriver,                    /* Handle to device  */
    (DWORD) IOCTL_BEST_NTMEM_UNMAP_USER_PHYSICAL_MEMORY,  /* IO Control code */
    &virt_addr,                 /* Buffer to driver. */
    sizeof(PVOID),              /* Length of buffer in bytes. */
    NULL,                       /* Buffer from driver.  */
    0,                          /* Length of buffer in bytes. */
    &cbReturned,                /* Bytes placed in OutBuffer. */
    NULL                        /* NULL means wait till op. completes. */
    );

  if (!IoctlResult)
  {
    print_err("DeviceIoControl failed (UNMAP_USER_PHYSICAL_MEMORY):\n%s\n");
  }
  /* keep driver open... if (!CloseHandle(hDriver)) { printf("CloseHandle
   * failed\n"); } */
end:;
}




// allocate a block of physical memory of given size
// returns 0 if successful, -1 if allocation fails
// virtual_addr and physical_addr contain the virtual resp. physical address of the
// allocated memory block (if the function succeeds)
int allocate_physical_memory(size_t size,
			     void **virtual_addr,
			     SPhysicalAddress *physical_addr)
{
  BOOL IoctlResult;
  ULONG ReturnedLength;
  MAP_MEM_ALLOC_MEM addresses;
  
  BYTE AddressDelta=0;   // offset to next 128 byte boundary
  UINT64 PhysAddress64=0;
  // chris:
  // We allocate 128 bytes more in order to be able to provide master with 
  // a physical address lying on a 128 byte boundary.
  // The returned memory is used as follows:
  // 1. 0-127 unused fill bytes (AddressDelta bytes) to reach ADB boundary for returned physical address
  // 2. One byte containing the number of used fill bytes from 1. (i.e. AddressDelta)
  // 3. m_size bytes for SVP test (pointer to these bytes is returned in physical address)
  // 4. 127-0 unused bytes

  size += ADB_MASK+1; // allocate 128 bytes more to be able to return an address lying on a 128 byte boundary

  addresses.Length = size;

  // Open driver
  if (hDriver == INVALID_HANDLE_VALUE)
    hDriver = CreateFile(
      BASE_CREATE_FILE_NAME_PCI,
      GENERIC_WRITE | GENERIC_READ,
      0,
      NULL,
      OPEN_EXISTING,
      FILE_ATTRIBUTE_NORMAL,
      NULL);

  if (hDriver == INVALID_HANDLE_VALUE)
  {
    print_err("CreateFile failed: %s\n");
    return -1;
  }

  /* Allocate buffer */
  IoctlResult = DeviceIoControl(
    hDriver,                                /* Handle to device  */
    IOCTL_BEST_NTMEM_ALLOC_PHYSICAL_MEMORY, /* IO Control code for Read */
    &addresses,                             /* Buffer to driver. */
    sizeof(addresses),                      /* Length of buffer in bytes. */
    &addresses,                             /* Buffer from driver.  */
    sizeof(addresses),                      /* Length of buffer in bytes. */
    &ReturnedLength,                        /* Bytes placed in DataBuffer. */
    NULL                                    /* NULL means wait till op. completes. */
    );

  if (!IoctlResult)
  {
    print_err("DeviceIoControl (IOCTL_BEST_NTMEM_ALLOC_PHYSICAL_MEMORY) failed:\n%s\n");
    return -1;
  }

  // Compute the one physical address
  PhysAddress64 = ((UINT64)addresses.physical_address.m_lowPart) |
                  (((UINT64)addresses.physical_address.m_highPart) << 32);

  PhysAddress64++; // one byte containing the skip length

  if (PhysAddress64 & ADB_MASK)
  {
    AddressDelta=(ADB_MASK+1)-((BYTE)(PhysAddress64 & ADB_MASK)); // in [1,..,127]
  }
  else
  {
    AddressDelta=0;
  }

  // Goto next ADB boundary
  PhysAddress64+=AddressDelta; 
  // PhysAddress64 is on a 128 byte boundary now
  
  // Convert back into two 32 bit integers
  addresses.physical_address.m_lowPart  = (unsigned long) (PhysAddress64 & (UINT64)0x00000000ffffffff);
  addresses.physical_address.m_highPart = (unsigned long) ((PhysAddress64 & (UINT64)0xffffffff00000000)>>32);

  // adapt virtual address correspondingly
  addresses.virtual_address = ((BYTE*)addresses.virtual_address)+AddressDelta+1; 

  // The byte before the returned address contains the AddressDelta
  ((BYTE*)addresses.virtual_address)[-1] = AddressDelta; // remember for later call of free_physical_memory()

  // Set return values
  *physical_addr = addresses.physical_address;
  *virtual_addr = addresses.virtual_address;

  return 0;
}

// free memory that has been allocated with allocate_physical_memory
// need to pass the virtual address and original size as parameters
// returns 0 if successful, otherwise -1
int free_physical_memory(void *virtual_addr )
{
  BOOL IoctlResult;
  MAP_MEM_ALLOC_MEM addr;
  ULONG ret_len;
  BYTE AddressDelta;
  
  // Open driver
  if (hDriver == INVALID_HANDLE_VALUE)
    hDriver = CreateFile(
      BASE_CREATE_FILE_NAME_PCI,
      GENERIC_WRITE | GENERIC_READ,
      0,
      NULL,
      OPEN_EXISTING,
      FILE_ATTRIBUTE_NORMAL,
      NULL);

  if (hDriver == INVALID_HANDLE_VALUE)
  {
    print_err("CreateFile failed: %s\n");
    return -1;
  }

  // chris
  // virtual_addr is from previous allocate_physical_memory()
  // We allocated 128 bytes more in order to be able to provide master with 
  // a physical address lying on a 128 byte boundary.
  // The allocated memory was used as follows:
  // 1. 0-127 unused fill bytes (AddressDelta bytes) to reach ADB boundary for returned physical address
  // 2. One byte containing the number of used fill bytes from 1. (i.e. AddressDelta)
  // 3. m_size bytes for SVP test (pointer to these bytes is returned in address)
  // 4. 127-0 unused bytes

  // virtual_addr is the address pointing to the m_size bytes from 3.

  AddressDelta=((BYTE*)(virtual_addr))[-1];
  
  // decrement by the amount of skip bytes and the one byte holding the number of skip bytes
  virtual_addr = ((BYTE*)virtual_addr)-AddressDelta-1;
  
  addr.virtual_address=virtual_addr;

  /* free memory */
  IoctlResult = DeviceIoControl(
    hDriver,                                /* Handle to device  */
    IOCTL_BEST_NTMEM_FREE_PHYSICAL_MEMORY,  /* IO Control code for Read */
    &addr,                                  /* Buffer to driver. */
    sizeof(addr),                           /* Length of buffer in bytes. */
    &addr,                                  /* Buffer from driver.  */
    0,                                      /* Length of buffer in bytes. */
    &ret_len,                               /* Bytes placed in DataBuffer. */
    NULL                                    /* NULL means wait till op. completes. */
    );

  if (!IoctlResult)
  {
    print_err("DeviceIoControl (FREE_PHYSICAL_MEMORY) failed:\n%s\n");
    return -1;
  }

  return 0;
}
 
/*****************************************************************************
 * The following two functions are only available for Windows 2000, and
 * are therefore "mapped" to the old ones for Win NT 4.0
 *****************************************************************************/

/*---------------------------------------------------------------------------*
 * int AllocatePhysicalMemory ()
 *
 * Purpose	: allocate memory using 
 *---------------------------------------------------------------------------*/
int AllocatePhysicalMemory ( SPhysicalMemory * pPhysicalMemory )
{
  return allocate_physical_memory(pPhysicalMemory->m_memSize,
			     &pPhysicalMemory->m_pVirtualAddr,
			     &pPhysicalMemory->m_physicalAddr);
}

/*---------------------------------------------------------------------------*
 * int DeAllocatePhysicalMemory (void * pVirtualAddr)
 *
 * Purpose	: deallocate memory reserved in AllocatePhysicalMemory64
 *---------------------------------------------------------------------------*/
int DeAllocatePhysicalMemory (SPhysicalMemory * pPhysicalMemory)
{
  // unmap
  return free_physical_memory(pPhysicalMemory->m_pVirtualAddr);
}

/*---------------------------------------------------------------------------*
 * int io_access(int cmd, unsigned long addr, unsigned long *data)
 *
 * Purpose	: read/write I/O ports with bytes/words/dwords
 *---------------------------------------------------------------------------*/
int io_access(int cmd, __int64 addr, unsigned long *data)
{
  BOOL IoctlResult;
  DWORD cbReturned;
  ULONG_PTR io_buf[3];
  if (hDriver == INVALID_HANDLE_VALUE)
    hDriver = CreateFile(
      BASE_CREATE_FILE_NAME_PCI,
      GENERIC_WRITE | GENERIC_READ,
      0,
      NULL,
      OPEN_EXISTING,
      FILE_ATTRIBUTE_NORMAL,
      NULL);

  if (hDriver == INVALID_HANDLE_VALUE)
  {
    print_err("CreateFile failed: %s\n");
    goto end;
  }

  io_buf[0] = cmd;
  io_buf[1] = (ULONG_PTR)addr;
  io_buf[2] = *data;

  IoctlResult = DeviceIoControl(
    hDriver,                    /* Handle to device  */
    (DWORD) IOCTL_BEST_NTMEM_IO_ACCESS, /* IO Control code */
    io_buf,                     /* Buffer to driver. */
    sizeof(io_buf),             /* Length of buffer in bytes. */
    io_buf,                     /* Buffer from driver.  */
    sizeof(io_buf),             /* Length of buffer in bytes. */
    &cbReturned,                /* Bytes placed in OutBuffer. */
    NULL                        /* NULL means wait till op. completes. */
    );

  if (!IoctlResult)
  {
    print_err("DeviceIoControl failed (IO_ACCESS):\n%s\n");
    return (-1);
  }

  *data = (unsigned long)io_buf[2];

  /* keep driver open... if (!CloseHandle(hDriver)) { printf("CloseHandle
   * failed\n"); } */
end:;
  return (0);
}

#if _WIN32_WINNT >= 0x0500
/*****************************************************************
   LoggedSetLockPagesPrivilege: a function to obtain, if possible, or
   release the privilege of locking physical pages.

   Inputs:

       HANDLE hProcess: Handle for the process for which we seek the
       privilege

       BOOL bEnable: Shall we enable (TRUE) or disable?

   Return value: TRUE indicates success, FALSE failure.

*****************************************************************/
static BOOL
LoggedSetLockPagesPrivilege ( HANDLE hProcess,
                              BOOL bEnable)
{
  struct {
    DWORD Count;
    LUID_AND_ATTRIBUTES Privilege [1];
  } Info;

  HANDLE Token;
  BOOL Result;

  // Open the token.

  Result = OpenProcessToken ( hProcess,
                              TOKEN_ADJUST_PRIVILEGES,
                              & Token);

  if( Result != TRUE ) {
    printf( "Cannot open process token.\n" );
    return FALSE;
  }

  // Enable or disable?

  Info.Count = 1;
  if( bEnable ) 
  {
    Info.Privilege[0].Attributes = SE_PRIVILEGE_ENABLED;
  } 
  else 
  {
    Info.Privilege[0].Attributes = 0;
  }

  // Get the LUID.

  Result = LookupPrivilegeValue ( NULL,
                                  SE_LOCK_MEMORY_NAME,
                                  &(Info.Privilege[0].Luid));

  if( Result != TRUE ) 
  {
    printf( "Cannot get privilege value for %s.\n", SE_LOCK_MEMORY_NAME );
    return FALSE;
  }

  // Adjust the privilege.

  Result = AdjustTokenPrivileges ( Token, FALSE,
                                   (PTOKEN_PRIVILEGES) &Info,
                                   0UL, NULL, NULL);

  // Check the result.

  if( Result != TRUE ) 
  {
    printf ("Cannot adjust token privileges, error %u.\n", GetLastError() );
    return FALSE;
  } 
  else 
  {
    print_err("Return code of AdjustTokenPrivileges: %s\n");
    if( GetLastError() != ERROR_SUCCESS ) 
    {
      printf ("Cannot enable SE_LOCK_MEMORY privilege, please check the local policy.\n");
      return FALSE;
    }
  }

  CloseHandle( Token );

  return TRUE;
}

#endif /* _WIN32_WINNT >= 0x0500 */

#ifdef MAKE_MAIN
int
  main()
#else
int
  test_physical_mem()
#endif
{
#define MAX_STR 10
  struct
  {
    void *aptr;
    SPhysicalAddress phys_addr;
    void *kern_addr;
    unsigned long size;
  } vstr[MAX_STR], astr[MAX_STR];
  int vcount = 0, acount = 0;
  int i, cont = 1;
  char buffer[100];
  while (cont)
  {
    printf("Select function (a=allocate, f=free, m=map_physical, u=unmap_physical_addr): ");
    gets(buffer);
    switch (*buffer)
    {
    case 'a':
      if (acount >= MAX_STR)
        printf("can't allocate more blocks\n");
      else
      {
        printf("Enter size of desired memory block: ");
        gets(buffer);
        astr[acount].size = strtoul(buffer, NULL, 0);
        if (allocate_physical_memory(astr[acount].size, &(astr[acount].aptr), &(astr[acount].phys_addr)))
        {
          printf("alloc failed\n");
        }
        else
        {
          printf("alloc successful:\n");
          printf("Virtual  address 0x%p\n", astr[acount].aptr);
          printf("Physical address 0x%08lx\n", astr[acount].phys_addr);
          printf("\n");
          acount++;
        }
      }
      break;

    case 'A':
      {
	SPhysicalMemory physMem;
        printf("Enter size of desired memory block: ");
        gets(buffer);
        physMem.m_memSize = strtoul(buffer, NULL, 0);
        if (AllocatePhysicalMemory(&physMem))
        {
          printf("AllocatePhysicalMemory failed\n");
        }
        else
        {
          printf("AllocatePhysicalMemory successful:\n");
          printf("Virtual  address 0x%p\n", physMem.m_pVirtualAddr);
          printf("Physical address 0x%08lx-%08lx\n", physMem.m_physicalAddr.m_highPart, physMem.m_physicalAddr.m_lowPart);
          printf("\n");

	  printf("Attempting deallocation:\n");
	  DeAllocatePhysicalMemory(&physMem);
        }
      }
      break;

    case 'f':
      if (acount == 0)
        printf("nothing to free\n");
      else
      {
        acount--;
        printf("freeing virtual address %p\n", astr[acount].aptr);
        free_physical_memory(astr[acount].aptr);
      }
      break;

    case 'm':
      if (vcount >= MAX_STR)
        printf("can't map more physical addresses\n");
      else
      {
        printf("Enter physical address: (only 32 bit addresses supported so far)");
        gets(buffer);
        vstr[vcount].phys_addr.m_lowPart = strtoul(buffer, NULL, 0);

        printf("Enter size: ");
        gets(buffer);
        vstr[vcount].size = strtoul(buffer, NULL, 0);

        vstr[vcount].aptr = map_physical_address(vstr[vcount].phys_addr.m_lowPart,0, vstr[vcount].size, 0, 0);
        if (vstr[vcount].aptr)
        {
          printf("Virtual  address 0x%p\n", vstr[vcount].aptr);
          printf("Data: ");
          for (i = 0; i < 16; i++)
            printf("%02x ", ((volatile unsigned char *) (vstr[vcount].aptr))[i]);
          printf("...\n");
          vcount++;
        }
        else
        {
          printf("map_physical_address failed\n");
        }
      }
      break;

    case 'u':
      if (vcount == 0)
      {
        printf("nothing to unmap\n");
      }
      else
      {
        vcount--;
        printf("unmapping virtual address %p\n", (vstr[vcount].aptr));
        unmap_physical_address(vstr[vcount].aptr);
      }
      break;

    case 'q':
      cont = 0;
      break;
    }
  }
  return (0);
}
#endif
